using System;
using System.IO;
using DynaPDF;

namespace signed_pdfa
{
   class Signed_PDFA
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // We open the output file later if no error occurrs.

            pdf.SetDocInfo(DynaPDF.TDocumentInfo.diCreator, "C# test application");
            pdf.SetDocInfo(DynaPDF.TDocumentInfo.diTitle, "Custom digital signature appearance");

            pdf.Append();
            pdf.SetFont("Arial", DynaPDF.TFStyle.fsNone, 10.0, true, DynaPDF.TCodepage.cp1252);
            pdf.WriteFText(DynaPDF.TTextAlign.taLeft, "This is a PDF/A 1b compatible PDF file that was digitally signed with " +
            "a self sign certificate. Because PDF/A requires that all fonts are embedded it is important " +
            "to avoid the usage of the 14 Standard fonts.\n\n" +

            "When signing a PDF/A compliant PDF file with the default settings (without creation of a user " +
            "defined appearance) the font Arial must be available on the system because it is used to print " +
            "the certificate properties into the signature field.\n\n" +

            "The font Arial must also be available if an empty signature field was added to the file " +
            "without signing it when closing the PDF file. Yes, it is still possible to sign a PDF/A " +
            "compliant PDF file later with Adobe's Acrobat. The signed PDF file is still compatible " +
            "to PDF/A. If you use a third party solution to digitally sign the PDF file then test " +
            "whether the signed file is still valid with the PDF/A 1b preflight tool included in Acrobat 8 " +
            "Professional.\n\n" +

            "Signature fields must be visible and the print flag must be set (default). CheckConformance() " +
            "adjusts these flags if necessary and produces a warning if changes were applied. If no changes " +
            "should be allowed, just return -1 in the error callback function. If the error callback function " +
            "returns 0, DynaPDF assumes that the prior changes were accepted and processing continues.\n\n" +

            "\\FC[255]Notice:\\FC[0]\n" +
            "It makes no sense to execute CheckConformance() without an error callback function or error event " +
            "in VB. If you cannot see what happens during the execution of CheckConformance(), it is " +
            "completely useless to use this function!\n\n" +

            "CheckConformance() should be used to find the right settings to create PDF/A compatible PDF files. " +
            "Once the the settings were found it is usually not longer recommended to execute this function. " +
            "However, it is of course possible to use CheckConformance() as a general approach to make sure " +
            "that files created with DynaPDF are PDF/A compatible.");

            /* ---------------------- Signature field appearance ---------------------- */

            int sigField = pdf.CreateSigField("Signature", -1, 200.0, 400.0, 200.0, 80.0);
            pdf.SetFieldColor(sigField, DynaPDF.TFieldColor.fcBorderColor, DynaPDF.TPDFColorSpace.csDeviceRGB, CPDF.NO_COLOR);
            // Place the validation icon on the left side of the signature field.
            pdf.PlaceSigFieldValidateIcon(sigField, 0.0, 15.0, 50.0, 50.0);
            /*
               This function creates a template into which you can draw whatever you want. The template
               is already opened when calling the function; it must be closed when finish with EndTemplate().
               An appearance template of a signature field is reserved for this field. It must not be placed
               on another page!

               In addition, it makes no sense to create an appearance template when the file is not digitally
               signed later. Empty signature fields cannot contain a user defined appearance.
            */
            pdf.CreateSigFieldAP(sigField);

            pdf.SaveGraphicState();
            pdf.Rectangle(0.0, 0.0, 200.0, 80.0, DynaPDF.TPathFillMode.fmNoFill);
            pdf.ClipPath(DynaPDF.TClippingMode.cmWinding, DynaPDF.TPathFillMode.fmNoFill);
            int sh = pdf.CreateAxialShading(0.0, 0.0, 200.0, 0.0, 0.5, pdf.RGB(120, 120, 220), CPDF.PDF_WHITE, true, true);
            pdf.ApplyShading(sh);
            pdf.RestoreGraphicState();

            pdf.SaveGraphicState();
            pdf.Ellipse(50.5, 1.0, 148.5, 78.0, DynaPDF.TPathFillMode.fmNoFill);
            pdf.ClipPath(DynaPDF.TClippingMode.cmWinding, DynaPDF.TPathFillMode.fmNoFill);
            sh = pdf.CreateAxialShading(0.0, 0.0, 0.0, 78.0, 2.0, CPDF.PDF_WHITE, pdf.RGB(120, 120, 220), true, true);
            pdf.ApplyShading(sh);
            pdf.RestoreGraphicState();

            pdf.SetFont("Arial", DynaPDF.TFStyle.fsBold | DynaPDF.TFStyle.fsUnderlined, 11.0, true, DynaPDF.TCodepage.cp1252);
            pdf.SetFillColor(pdf.RGB(120, 120, 220));
            pdf.WriteFTextEx(50.0, 60.0, 150.0, -1.0, DynaPDF.TTextAlign.taCenter, "Digitally signed by:");
            pdf.SetFont("Arial", DynaPDF.TFStyle.fsBold | DynaPDF.TFStyle.fsItalic, 18.0, true, DynaPDF.TCodepage.cp1252);
            pdf.SetFillColor(pdf.RGB(100, 100, 200));
            pdf.WriteFTextEx(50.0, 45.0, 150.0, -1.0, DynaPDF.TTextAlign.taCenter, "DynaPDF");

            pdf.EndTemplate(); // Close the appearance template.

            /* ------------------------------------------------------------------------ */

            pdf.EndPage();
            // Check whether the file is compatible to PDF/A 1b
            switch (pdf.CheckConformance(DynaPDF.TConformanceType.ctPDFA_1b_2005, 0, IntPtr.Zero, null, null))
            {
               case 1: // RGB
               case 3: // Gray
                  pdf.AddOutputIntent("../../../test_files/sRGB.icc");
                  break;
               case 2: // CMYK
                  pdf.AddOutputIntent("../../../test_files/ISOcoated_v2_bas.ICC");
                  break;
               default: break;
            }
            // No fatal error occurred?
            if (pdf.HaveOpenDoc())
            {
               // We write the output file into the current directory.
               String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
               // OK, now we can open the output file.
               if (!pdf.OpenOutputFile(filePath)) return;
               if (pdf.CloseAndSignFile("../../../test_files/test_cert.pfx", "123456", "Test", null))
               {
                  Console.Write("PDF file \"{0}\" successfully created!\n", filePath);
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = filePath;
                  p.Start();
               }
            }
            pdf = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
         }
         Console.Read();
		}
	}
}
